<?php

namespace App\Http\Controllers\Storefront;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Support\CartOwnership;
use App\Support\CouponAllocator;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

class AuthController extends Controller
{
    public function showLoginForm(): View
    {
        return view('pages.login');
    }

    public function showRegistrationForm(): View
    {
        return view('pages.signup');
    }

    public function login(Request $request): RedirectResponse
    {
        $credentials = $request->validate([
            'email' => ['required', 'string', 'email'],
            'password' => ['required', 'string'],
        ]);

        $remember = $request->boolean('remember');
        $sessionId = $request->session()->getId();

        if (! Auth::attempt($credentials, $remember)) {
            return back()->withErrors([
                'email' => 'We could not sign you in with those credentials.',
            ])->onlyInput('email');
        }

        $request->session()->regenerate();

        $authUser = Auth::user();

        CartOwnership::migrateSessionCart($sessionId, $authUser);

        $authUser?->forceFill([
            'last_login_at' => now(),
        ])->save();

        if ($authUser) {
            app(CouponAllocator::class)->assignWelcomeBundle($authUser);
        }

        return redirect()->intended(route('account.dashboard'));
    }

    public function register(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email'],
            'password' => ['required', 'confirmed', Password::defaults()],
        ]);

        $sessionId = $request->session()->getId();

        $user = User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
            'is_admin' => false,
            'role' => 'customer',
        ]);

        Auth::login($user);

        $request->session()->regenerate();

        $user->forceFill([
            'last_login_at' => now(),
        ])->save();

        CartOwnership::migrateSessionCart($sessionId, $user);

        app(CouponAllocator::class)->assignWelcomeBundle($user);

        return redirect()->route('account.dashboard');
    }

    public function logout(Request $request): RedirectResponse
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('home');
    }
}
